/**
 * A pong game using Arduino and Processing
 *
 * This example demonstrates the use of Processing to create animations as well as the 
 * use of the 'serial' package to read-in values from a serial port. In particular, 
 * these values come from an Arduino board and correspond to the position (resistance value)
 * of a potentiometer. We use this value (pot) to update the position of a paddle on our
 * screen.
 *
 * Spyros Veronikis (spyros.veronikis@gmail.com), March 2013
 */

import processing.serial.*;
Serial port;

// Declare GLOBAL variables
int padW; int padH; 
float ypos;
float ball_x; float ball_y; float ballvel_x; float ballvel_y; int diameter;

boolean play;
int goals; int gap;
int pot;  // Use a variable to store the potentiometer value

// User-defined functions 
void ball_init(){
  ball_x = int(width/2);    ball_y = int(height/2);   // Set the ball position at start
  // Set the initial ball velocity
  ballvel_x = int(random(0.2, 0.5)*10);  ballvel_y = int(random(0.2, 0.3)*10); 
}

// Initialization
void setup(){
  size(600, 400);                        // Set the frame size
  ellipseMode(CENTER); smooth();         // Initialize graphics settings
  // Initialize Serial Communication interface (set port number and data rate)
  String arduinoPort = Serial.list()[7];
  port = new Serial(this, arduinoPort, 9600);
  goals = 0; play = true;               // Set the initial game values
  gap = 50;
  padW = 10; padH = 50;                 // Initialize paddle values
  ball_init();  diameter = 30;          // Initialize the ball position and diameter
}

// Main program
void draw(){
  background(30);
  textSize(12);  
  text("Goals: ", 120, 20); text(goals, 170, 20);
  stroke(204, 102, 0); line(width/2, 0, width/2, height);
  strokeWeight(2); noFill(); ellipse(width/2, height/2, 100, 100);
  fill(255);  noStroke();

  if(play){   
    // Update ball position
    ball_x = ball_x + ballvel_x;
    ball_y = ball_y + ballvel_y;

    // Reflect off bottom and top walls
    if ((ball_y >= height - diameter/2) || (ball_y < 0+diameter/2)){
        ballvel_y = -ballvel_y; }

    if (ball_x >= width - diameter/2){
        ballvel_x = -ballvel_x;  } 

    if (ball_x < gap + padW + diameter/2){
      if ((abs((ypos+padH/2)-ball_y ) < 0.75*padH) && (ball_x > gap+padW-3) ){
        ballvel_x = -1.20 * ballvel_x;    // That's a physics law violation! (increase speed 20%)
      }
    }
    if (ball_x < 0 ){   play = false;    // Stop game 
                        goals++;         // Increment goals taken
                      }

    ellipse(ball_x, ball_y, diameter, diameter);  // Draw the ball
    
    // Get the paddle position from potentiometer
    if(port.available() > 0){
      if (port.read() == 0xff){pot = (port.read() << 8) | (port.read());}
    }
    if (pot != -1) {           // We have valid data from potentiometer
      ypos = pot;              // Read-in the data
      ypos = map(ypos, 0, 511, 0, height-padH);  // Map the data to a corresponding scale
    }
    
  } else {  textSize(36);            // if game is not in play...
            text("Press 'c' to continue", 120, 200);
            text("or 'r' to restart!", 120, 240);
         }
         
  if (keyPressed){
    switch(key){
      case 'c':
        play = true; ball_init(); break;
      case 'r':
        play = true; ball_init(); goals = 0; break;
    }
  }
  // Finally, draw the paddle
  rect(gap, ypos, 10, 50);  
}
